const xpath = require("./utils/xpath.js");
const DOMParser = require("./utils/dom.js");

class Cheerio {
    constructor() {
        this.document = null;
        this.nodes = [];
    }

    load(htmlString) {
        const { document, html: normalizeHtml } = DOMParser(htmlString);
        this.document = document;
        this.nodes = [document];

        // 辅助函数：包装 Cheerio 实例，返回可链式调用的 $ 函数
        const createWrapper = instance => {
            const wrapper = selector => {
                if (selector) {
                    return instance.find(selector);
                }
                return wrapper;
            };

            // 标记该实例为 Cheerio 包装实例，避免重复包装
            wrapper.__isCheerio = true;

            // 绑定 Cheerio 原型上的方法（排除 constructor 和 load）
            Object.getOwnPropertyNames(Cheerio.prototype).forEach(method => {
                if (method !== "constructor" && method !== "load") {
                    wrapper[method] = Cheerio.prototype[method].bind(instance);
                }
            });

            // 直接绑定方法，支持链式调用
            wrapper.find = selector => instance.find(selector);
            wrapper.text = () => instance.text();
            wrapper.each = callback => instance.each(callback);
            wrapper.attr = (name, value) => instance.attr(name, value);
            wrapper.last = () => instance.last();
            wrapper.first = () => instance.first();
            wrapper.eq = index => instance.eq(index);

            // 绑定原始节点和 document
            wrapper.nodes = instance.nodes;
            wrapper.document = instance.document;

            return wrapper;
        };

        // 定义 $ 方法，支持不同类型的参数
        const $ = selector => {
            // 如果已经是 Cheerio 实例，直接返回，避免重复包装
            if (
                selector &&
                typeof selector === "object" &&
                selector.__isCheerio
            ) {
                return selector;
            }
            // 如果传入的是 DOM 节点，则包装成新的 Cheerio 实例
            if (
                selector &&
                typeof selector === "object" &&
                "nodeType" in selector
            ) {
                const newInstance = new Cheerio();
                newInstance.document = this.document;
                newInstance.nodes = [selector];
                return createWrapper(newInstance);
            }
            // 如果传入的是整个文档对象，则返回当前实例
            else if (selector === document) {
                return createWrapper(this);
            }
            // 如果是字符串，则执行 find 方法
            else if (typeof selector === "string") {
                return this.find(selector);
            }
            // 默认返回当前实例的包装器
            return createWrapper(this);
        };

        // 返回 $ 以供使用
        return $;
    }

    find(selector) {
        if (!this.document) {
            throw new Error("No document loaded. Use load() first.");
        }

        const xpathExpression = this.cssToXPath(selector);
        const newInstance = new Cheerio();
        newInstance.document = this.document;
        newInstance.nodes = [];

        this.nodes.forEach(node => {
            // 适配相对路径查询
            let query =
                node.nodeType === 9
                    ? xpathExpression
                    : xpathExpression.startsWith("//")
                      ? "." + xpathExpression
                      : xpathExpression;
            const foundNodes = xpath(node, query);
            newInstance.nodes.push(...foundNodes);
        });

        // 创建包装器
        const $ = sel => {
            if (sel) {
                return newInstance.find(sel);
            }
            return $;
        };

        Object.getOwnPropertyNames(Cheerio.prototype).forEach(method => {
            if (method !== "constructor" && method !== "load") {
                $[method] = Cheerio.prototype[method].bind(newInstance);
            }
        });
        $.find = selector => newInstance.find(selector);
        $.text = () => newInstance.text();
        $.each = callback => newInstance.each(callback);
        $.attr = name => newInstance.attr(name);
        $.last = () => newInstance.last();
        $.first = () => newInstance.first();
        $.eq = index => newInstance.eq(index);

        $.nodes = newInstance.nodes;
        $.document = newInstance.document;
        return $;
    }

    cssToXPath(selector) {
        // 处理逗号分隔的多个选择器
        if (selector.includes(",")) {
            const selectors = selector.split(",").map(s => s.trim());
            return selectors.map(s => this.cssToXPath(s)).join(" | ");
        }

        // 解析复杂选择器
        let parts = [];
        let currentPart = "";
        let inAttribute = false;

        for (let i = 0; i < selector.length; i++) {
            const char = selector[i];

            if (char === "[") inAttribute = true;
            if (char === "]") inAttribute = false;

            if (
                !inAttribute &&
                (char === " " || char === ">" || char === "+" || char === "~")
            ) {
                if (currentPart.trim()) {
                    parts.push({
                        selector: currentPart.trim(),
                        combinator: char
                    });
                    currentPart = "";
                }
            } else {
                currentPart += char;
            }
        }

        if (currentPart.trim()) {
            parts.push({
                selector: currentPart.trim(),
                combinator: null
            });
        }

        let xpath = "";
        let contextPath = "";

        for (let i = 0; i < parts.length; i++) {
            const { selector, combinator } = parts[i];
            const prevCombinator = i > 0 ? parts[i - 1].combinator : null;

            // 处理基本选择器
            let selectorXPath = this.simpleToXPath(selector);

            // 应用组合符
            if (i === 0) {
                xpath = "//" + selectorXPath;
                contextPath = selectorXPath;
            } else {
                switch (prevCombinator) {
                    case " ": // 后代选择器
                        xpath += "//" + selectorXPath;
                        contextPath += "//" + selectorXPath;
                        break;
                    case ">": // 子元素选择器
                        xpath += "/" + selectorXPath;
                        contextPath += "/" + selectorXPath;
                        break;
                    case "+": // 相邻兄弟选择器
                        xpath +=
                            "/following-sibling::*[1][self::" +
                            selectorXPath +
                            "]";
                        contextPath =
                            "following-sibling::*[1][self::" +
                            selectorXPath +
                            "]";
                        break;
                    case "~": // 一般兄弟选择器
                        xpath += "/following-sibling::" + selectorXPath;
                        contextPath = "following-sibling::" + selectorXPath;
                        break;
                }
            }
        }

        return xpath;
    }

    simpleToXPath(selector) {
        // 处理 ID 选择器
        if (selector.startsWith("#")) {
            return `*[@id='${selector.slice(1)}']`;
        }

        // 处理类选择器
        if (selector.startsWith(".")) {
            return `*[contains(concat(' ', normalize-space(@class), ' '), ' ${selector.slice(1)} ')]`;
        }

        // 处理属性选择器（例如 a[href="http://"]）
        const attrMatch = selector.match(/^([a-zA-Z0-9]+)?\[(.+)\]$/);
        if (attrMatch) {
            const tag = attrMatch[1] || "*"; // 可能是 `img[src]` 这种，tag 可选
            const attrSelector = attrMatch[2];

            // 解析属性选择器
            const attrParts = attrSelector.match(
                /^([^=~|^$*]+)(?:([=~|^$*]+)['"]?([^'"\]]+)['"]?)?$/
            );
            if (attrParts) {
                const [, attrName, operator, value] = attrParts;
                switch (operator) {
                    case "=":
                        return `${tag}[@${attrName}='${value}']`;
                    case "~=":
                        return `${tag}[contains(concat(' ', normalize-space(@${attrName}), ' '), ' ${value} ')]`;
                    case "|=":
                        return `${tag}[@${attrName}='${value}' or starts-with(@${attrName}, '${value}-')]`;
                    case "^=":
                        return `${tag}[starts-with(@${attrName}, '${value}')]`;
                    case "$=":
                        return `${tag}[substring(@${attrName}, string-length(@${attrName}) - string-length('${value}') + 1) = '${value}']`;
                    case "*=":
                        return `${tag}[contains(@${attrName}, '${value}')]`;
                    default:
                        return `${tag}[@${attrName}]`; // 只匹配存在该属性的元素
                }
            }
        }

        // 处理通配符 `*` 选择器 或 普通标签选择器
        return selector === "*" ? "*" : selector;
    }

    each(callback) {
        // 遍历原始节点，不对节点提前包装
        this.nodes.forEach((node, index) => {
            // 直接传入原始节点，这样在回调中可以通过 $(item) 进行包装
            callback.call(node, index, node);
        });
        return this;
    }

    createNodeWrapper() {
        const self = this;
        return function (node) {
            if (node && typeof node === "object" && "nodeType" in node) {
                const newInstance = new Cheerio();
                newInstance.document = self.document;
                newInstance.nodes = [node];

                // Create a wrapper function with Cheerio methods
                const $ = selector => {
                    if (selector) {
                        return newInstance.find(selector);
                    }
                    return $;
                };

                // Copy all prototype methods to the function
                Object.getOwnPropertyNames(Cheerio.prototype).forEach(
                    method => {
                        if (method !== "constructor" && method !== "load") {
                            $[method] =
                                Cheerio.prototype[method].bind(newInstance);
                        }
                    }
                );

                $.find = selector => newInstance.find(selector);
                $.text = () => newInstance.text();
                $.each = callback => newInstance.each(callback);
                $.attr = (name, value) => newInstance.attr(name, value);
                $.last = () => newInstance.last();
                $.first = () => newInstance.first();
                $.eq = index => newInstance.eq(index);

                $.nodes = newInstance.nodes;
                $.document = newInstance.document;

                return $;
            }
            return null;
        };
    }

    attr(name) {
        // 只需要实现读
        if (this.nodes.length === 0) {
            return undefined;
        }
        return this.nodes[0].getAttribute(name);
    }

    last() {
        if (this.nodes.length === 0) {
            return new Cheerio(); // 返回空实例
        }
        const lastNode = this.nodes[this.nodes.length - 1]; // 获取最后一个元素
        const newInstance = new Cheerio();
        newInstance.document = this.document;
        newInstance.nodes = [lastNode];
        return newInstance;
    }

    first() {
        if (this.nodes.length === 0) {
            return new Cheerio(); // 返回空实例
        }
        const firstNode = this.nodes[0]; // 获取第一个元素
        const newInstance = new Cheerio();
        newInstance.document = this.document;
        newInstance.nodes = [firstNode];
        return newInstance;
    }

    eq(index) {
        if (
            this.nodes.length === 0 ||
            index < 0 ||
            index >= this.nodes.length
        ) {
            return new Cheerio(); // 返回空实例
        }
        const eqNode = this.nodes[index]; // 获取指定索引的元素
        const newInstance = new Cheerio();
        newInstance.document = this.document;
        newInstance.nodes = [eqNode];
        return newInstance;
    }

    text() {
        return this.nodes.map(node => node.textContent.trim()).join(", ");
    }
}

module.exports = Cheerio;
