const { deepMerge, unescapeEntity } = require("./utils/index.js");
const { request: http, success, error } = require("./utils/request.js");
const xpath = require("./utils/xpath.js");
const DOMParser = require("./utils/dom.js");

const config = globalThis.config;

const direction = {
    horizontal: "horizontal",
    vertical: "vertical"
};

function request(url, options) {
    const ua = config?.ua?.trim();

    const defaultOptions = {
        headers: {}
    };
    if (ua) {
        defaultOptions.headers["user-agent"] = ua;
    }

    return http(url, deepMerge(defaultOptions, options));
}

/**
 * 解析xpath字段并匹配对应的regexp规则
 * @param {*} dom
 * @param {*} field
 * @returns
 */
function parseXpathText(dom, field) {
    let text = "";
    try {
        if (!config[field]) return "";

        const node = xpath(dom, config[field]);
        text = node.map(item => item.nodeValue?.trim()).join(",");

        if (config[`${field}_regexp`]) {
            const matchVal = text.match(new RegExp(config[`${field}_regexp`]));
            if (matchVal) {
                text = matchVal[1];
            } else {
                text = "";
            }
        }
        return text;
    } catch (e) {
        console.error(e);
    }
    return text;
}

globalThis.DOMParser = DOMParser;
globalThis.xpath = xpath;
globalThis.$fetch = request;
globalThis.parseXpathText = parseXpathText;

async function home() {
    try {
        if (globalThis.ext && config.home_function) {
            const result = await globalThis.ext[config.home_function]();
            return success(result);
        }
        const response = await request(config.home_url);
        const html = await response.text();

        const document = DOMParser(html);

        const $sections = xpath(document, config.home_cate_node);
        const result = [];

        $sections.forEach(($row, i) => {
            const type_name = parseXpathText($row, "home_cate_name");
            const type_id = parseXpathText($row, "home_cate_id");

            const vod_list = [];

            const $items = xpath($row, config.home_vod_node);
            $items.forEach($item => {
                const vod_id = parseXpathText($item, "home_vod_id");
                const vod_name = parseXpathText($item, "home_vod_name");
                const vod_pic = parseXpathText($item, "home_vod_pic");

                const vod_remarks = parseXpathText($item, "home_vod_remarks");
                const vod_preview = parseXpathText($item, "home_vod_preview");

                if (vod_id) {
                    vod_list.push({
                        type_id,
                        vod_id,
                        vod_name,
                        vod_pic,
                        vod_remarks,
                        vod_preview,
                        direction:
                            config?.home_vod_direction || direction.vertical
                    });
                }
            });
            if (vod_list.length) {
                result.push({
                    type_id,
                    type_name,
                    vod_list
                });
            }
        });
        return success(result);
    } catch (e) {
        console.error(e);
        return error(e.message);
    }
}

async function detail(params) {
    try {
        const { vod_id } = params;
        if (globalThis.ext && config.detail_function) {
            const result = await globalThis.ext[config.detail_function](params);
            return success(result);
        }
        const url = config.detail_url.replace(/{vod_id}/g, vod_id);

        const response = await request(url);
        const html = await response.text();
        const document = DOMParser(html);

        const vod_actor = parseXpathText(document, "detail_vod_actor");
        const vod_area = parseXpathText(document, "detail_vod_area");
        const vod_class = parseXpathText(document, "detail_vod_class");
        const vod_lang = parseXpathText(document, "detail_vod_lang");
        const vod_time = parseXpathText(document, "detail_vod_time");
        const vod_year = parseXpathText(document, "detail_vod_year");
        const vod_remarks = parseXpathText(document, "detail_vod_remarks");

        const vod_content = parseXpathText(document, "detail_vod_content");

        const vod_name = parseXpathText(document, "detail_vod_name");
        const vod_pic = parseXpathText(document, "detail_vod_pic");

        const vod_sources = [];

        const $sources = xpath(document, config.detail_source_node);
        for (let i = 0, len = $sources.length; i < len; i++) {
            const $source = $sources[i];

            const source_name = parseXpathText($source, "detail_source_name");

            if (source_name) {
                const source = {
                    source_name: source_name,
                    vod_play_list: {
                        url_count: 0,
                        urls: []
                    }
                };
                const $links = xpath($source, config.detail_url_node);
                source.vod_play_list.url_count = $links.length;
                $links.forEach($link => {
                    const url = {
                        name: parseXpathText($link, "detail_url_name"),
                        url: parseXpathText($link, "detail_url_id")
                    };
                    source.vod_play_list.urls.push(url);
                });
                $links.length && vod_sources.push(source);
            }
        }

        const result = {
            vod_actor,
            vod_area,
            vod_class,
            vod_content,
            vod_id,
            vod_lang,
            vod_name,
            vod_pic,
            vod_time,
            vod_year,
            vod_remarks,
            vod_sources,
            similar: similar(document)
        };
        return success(result);
    } catch (e) {
        console.error(e);
        return error(e.message);
    }
}

async function list(type_id, page = 1) {
    try {
        if (globalThis.ext && config.list_function) {
            const result = await globalThis.ext[config.list_function](
                type_id,
                page
            );
            return success(result);
        }
        let url = config.list_url;
        // 是否有配置规则
        // 如果有配置规则的话，替换掉list_url
        if (config.list_manual) {
            if (config.list_manual?.[type_id]?.url) {
                url = config.list_manual[type_id].url;
            } else if (config.list_manual?.["*"]?.url) {
                url = config.list_manual["*"].url;
            }
        }
        const fetchUrl = url
            .replace(/{type_id}/g, type_id)
            .replace(/{page}/g, page);

        console.log(fetchUrl);
        const response = await request(fetchUrl);
        const html = await response.text();
        const document = DOMParser(html);

        const list = [];
        const $sections = xpath(document, config.list_vod_node);
        $sections.forEach($row => {
            const vod_name = parseXpathText($row, "list_vod_name");
            const vod_remarks = parseXpathText($row, "list_vod_remarks");
            const vod_pic = parseXpathText($row, "list_vod_pic");
            const vod_id = parseXpathText($row, "list_vod_id");
            const vod_preview = parseXpathText($row, "list_vod_preview");
            if (vod_id) {
                list.push({
                    vod_id,
                    vod_name,
                    vod_pic,
                    vod_remarks,
                    vod_preview,
                    direction: config?.list_vod_direction || direction.vertical
                });
            }
        });

        let pages = 1;
        if (config.list_page_node) {
            const $pages = xpath(document, config.list_page_node)[0];
            if ($pages) {
                pages = parseXpathText($pages, "list_page_id");
                if (pages) {
                    pages = Number(pages);
                }
            }
        }
        const result = {
            pages,
            page,
            list
        };

        return success(result);
    } catch (e) {
        console.error(e);
        return error(e.message);
    }
}

async function search(keyword, page = 1) {
    try {
        if (globalThis.ext && config.search_function) {
            const result = await globalThis.ext[config.search_function](
                keyword,
                page
            );
            return success(result);
        }
        const fetchUrl = config.search_url
            .replace(/{wd}/g, keyword)
            .replace(/{page}/g, page);
        const response = await request(fetchUrl);
        const html = await response.text();
        const document = DOMParser(html);

        const list = [];
        const $sections = xpath(document, config.search_vod_node);
        $sections.forEach($row => {
            const vod_name = parseXpathText($row, "search_vod_name");
            const vod_remarks = parseXpathText($row, "search_vod_remarks");
            const vod_pic = parseXpathText($row, "search_vod_pic");
            const vod_id = parseXpathText($row, "search_vod_id");
            const vod_preview = parseXpathText($row, "search_vod_preview");

            if (vod_id) {
                list.push({
                    vod_id,
                    vod_name,
                    vod_pic,
                    vod_remarks,
                    vod_preview,
                    direction:
                        config?.search_vod_direction || direction.vertical
                });
            }
        });

        let pages = 1;
        if (config.list_page_node) {
            const $pages = xpath(document, config.search_page_node)[0];
            if ($pages) {
                pages = parseXpathText($pages, "search_page_id");
                if (pages) {
                    pages = Number(pages);
                }
            }
        }
        const result = {
            pages,
            page,
            list
        };
        return success(result);
    } catch (e) {
        console.error(e);
        return error(e.message);
    }
}

function similar(document) {
    try {
        if (!config.similar_node) return [];
        const $list = xpath(document, config.similar_node);
        const list = [];
        if ($list.length) {
            [].forEach.call($list ?? [], (node, i) => {
                const vod_name = parseXpathText(node, "similar_vod_name");
                const vod_pic = parseXpathText(node, "similar_vod_pic");
                const vod_remarks = parseXpathText(node, "similar_vod_remarks");
                const vod_id = parseXpathText(node, "similar_vod_id");
                const vod_preview = parseXpathText(node, "similar_vod_preview");

                if (vod_id) {
                    list.push({
                        vod_id,
                        vod_name,
                        vod_pic,
                        vod_remarks,
                        vod_preview,
                        direction:
                            config?.similar_vod_direction || direction.vertical
                    });
                }
            });
        }
        return list;
    } catch (e) {
        return [];
    }
}

/**
 * 这里传入的是分集id, 不是视频id
 * @param {*} url  "/ep-449073-17-1.html"
 */
async function play({ url }) {
    try {
        if (globalThis.ext && config.play_function) {
            const playUrl = await globalThis.ext[config.play_function](url);
            return success(playUrl);
        } else {
            return success(url);
        }
    } catch (e) {
        console.error(e);
        return error(e.message);
    }
}

module.exports = {
    home,
    detail,
    play,
    search,
    list
};
